/**
 * Script de génération du communiqué de presse LawCodex
 * Génère : HTML, PDF (via Puppeteer), DOCX (via docx)
 */

const fs = require('fs');
const path = require('path');
const puppeteer = require('puppeteer');
const { Document, Packer, Paragraph, TextRun, HeadingLevel, AlignmentType, BorderStyle, ImageRun, ExternalHyperlink, Table, TableRow, TableCell, WidthType, ShadingType, TableBorders } = require('docx');

const DIR = path.resolve(__dirname);
const LOGO_PATH = path.join(DIR, 'logos_lawcodex', 'logo_defaut_primary.png');
const LOGO_WHITE_PATH = path.join(DIR, 'logos_lawcodex', 'logo_defaut_primary-white.png');
const logoWhiteBase64 = fs.readFileSync(LOGO_WHITE_PATH).toString('base64');
const logoBuffer = fs.readFileSync(LOGO_PATH);

// LawCodex brand colors
const COLORS = {
  navyBlue: '#1B2A5B',
  darkRed: '#8B1A1A',
  gray: '#6B7280',
  lightGray: '#F3F4F6',
  white: '#FFFFFF',
  black: '#111827',
};

// ────────────────────────────────────────
// 1. HTML
// ────────────────────────────────────────
function generateHTML() {
  const html = `<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Communiqué de presse — LawCodex — Lancement 5 mars 2026</title>
  <style>
    @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;600;700&family=Inter:wght@300;400;500;600;700&display=swap');

    :root {
      --navy: ${COLORS.navyBlue};
      --red: ${COLORS.darkRed};
      --gray: ${COLORS.gray};
      --light-gray: ${COLORS.lightGray};
      --white: ${COLORS.white};
      --black: ${COLORS.black};
    }

    * { margin: 0; padding: 0; box-sizing: border-box; }

    body {
      font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
      color: var(--black);
      line-height: 1.7;
      background: #fff;
    }

    .container {
      max-width: 780px;
      margin: 0 auto;
      padding: 0 32px;
    }

    /* Header */
    .header {
      background: linear-gradient(135deg, var(--navy) 0%, #2d3f7a 100%);
      padding: 40px 0 36px;
      text-align: center;
    }
    .header img.logo {
      height: 90px;
      max-width: 220px;
      width: auto;
      margin: 0 auto 20px;
      display: block;
    }
    .header .badge {
      display: inline-block;
      background: var(--red);
      color: #fff;
      font-size: 11px;
      font-weight: 600;
      letter-spacing: 2px;
      text-transform: uppercase;
      padding: 6px 20px;
      border-radius: 20px;
      margin-bottom: 12px;
    }
    .header h1 {
      font-family: 'Playfair Display', Georgia, serif;
      color: #fff;
      font-size: 26px;
      font-weight: 700;
      line-height: 1.3;
      margin-bottom: 10px;
      padding: 0 20px;
    }
    .header .subtitle {
      color: rgba(255,255,255,0.85);
      font-size: 14px;
      font-weight: 400;
      max-width: 600px;
      margin: 0 auto;
    }

    /* Meta bar */
    .meta-bar {
      background: var(--light-gray);
      border-bottom: 3px solid var(--red);
      padding: 14px 0;
      text-align: center;
      font-size: 13px;
      color: var(--gray);
    }
    .meta-bar strong { color: var(--navy); }

    /* Content */
    .content { padding: 40px 0 20px; }

    .content .lead {
      font-size: 16px;
      line-height: 1.8;
      color: #374151;
      margin-bottom: 32px;
      padding-bottom: 24px;
      border-bottom: 1px solid #e5e7eb;
    }

    h2 {
      font-family: 'Playfair Display', Georgia, serif;
      color: var(--navy);
      font-size: 22px;
      font-weight: 700;
      margin: 36px 0 16px;
      padding-bottom: 8px;
      border-bottom: 2px solid var(--red);
      display: inline-block;
    }

    h2::after {
      content: '';
      display: block;
    }

    p {
      margin-bottom: 16px;
      font-size: 14.5px;
      color: #374151;
    }

    .quote-block {
      background: linear-gradient(135deg, #f8f4f0 0%, #fdf9f6 100%);
      border-left: 4px solid var(--red);
      border-radius: 0 8px 8px 0;
      padding: 24px 28px;
      margin: 28px 0;
    }
    .quote-block .quote-text {
      font-family: 'Playfair Display', Georgia, serif;
      font-style: italic;
      font-size: 15px;
      color: var(--navy);
      line-height: 1.7;
      margin-bottom: 10px;
    }
    .quote-block .quote-author {
      font-size: 13px;
      font-weight: 600;
      color: var(--red);
    }

    /* Features list */
    .features-list {
      list-style: none;
      padding: 0;
      margin: 16px 0 28px;
    }
    .features-list li {
      padding: 10px 0 10px 24px;
      position: relative;
      font-size: 14px;
      color: #374151;
      border-bottom: 1px solid #f3f4f6;
    }
    .features-list li::before {
      content: '■';
      position: absolute;
      left: 0;
      color: var(--red);
      font-size: 10px;
      top: 13px;
    }
    .features-list li strong {
      color: var(--navy);
    }

    /* Table */
    .plans-table {
      width: 100%;
      border-collapse: collapse;
      margin: 20px 0 28px;
      font-size: 13.5px;
    }
    .plans-table th {
      background: var(--navy);
      color: #fff;
      padding: 12px 16px;
      text-align: left;
      font-weight: 600;
      font-size: 13px;
    }
    .plans-table td {
      padding: 12px 16px;
      border-bottom: 1px solid #e5e7eb;
      color: #374151;
      vertical-align: top;
    }
    .plans-table tr:nth-child(even) td {
      background: #f9fafb;
    }
    .plans-table td:first-child {
      font-weight: 600;
      color: var(--navy);
    }

    /* Key figures */
    .key-figures {
      display: flex;
      gap: 16px;
      flex-wrap: wrap;
      margin: 20px 0 32px;
    }
    .key-figure {
      background: var(--light-gray);
      border-radius: 10px;
      padding: 18px 24px;
      text-align: center;
      flex: 1;
      min-width: 120px;
    }
    .key-figure .number {
      font-family: 'Playfair Display', Georgia, serif;
      font-size: 28px;
      font-weight: 700;
      color: var(--red);
      display: block;
    }
    .key-figure .label {
      font-size: 12px;
      color: var(--gray);
      margin-top: 4px;
    }

    /* Launch box */
    .launch-box {
      background: linear-gradient(135deg, var(--navy) 0%, #2d3f7a 100%);
      border-radius: 12px;
      padding: 32px;
      text-align: center;
      margin: 32px 0;
      color: #fff;
    }
    .launch-box h3 {
      font-family: 'Playfair Display', Georgia, serif;
      font-size: 22px;
      margin-bottom: 8px;
    }
    .launch-box .launch-date {
      font-size: 20px;
      font-weight: 700;
      color: #fbbf24;
      margin: 8px 0;
    }
    .launch-box .launch-url {
      font-size: 16px;
      color: rgba(255,255,255,0.9);
    }
    .launch-box .launch-url a {
      color: #93c5fd;
      text-decoration: none;
      font-weight: 600;
    }

    /* About box */
    .about-box {
      background: #fafafa;
      border: 1px solid #e5e7eb;
      border-radius: 10px;
      padding: 28px;
      margin: 28px 0;
    }
    .about-box h3 {
      font-family: 'Playfair Display', Georgia, serif;
      color: var(--navy);
      font-size: 18px;
      margin-bottom: 12px;
    }
    .about-box p {
      font-size: 14px;
      color: #4b5563;
    }
    .about-box .links {
      margin-top: 16px;
      font-size: 13px;
    }
    .about-box .links a {
      color: var(--red);
      text-decoration: none;
      margin-right: 20px;
      font-weight: 500;
    }

    /* Contact */
    .contact-box {
      background: var(--light-gray);
      border-radius: 10px;
      padding: 24px 28px;
      margin: 24px 0 40px;
    }
    .contact-box h3 {
      color: var(--navy);
      font-size: 16px;
      margin-bottom: 12px;
    }
    .contact-box p {
      font-size: 13.5px;
      margin-bottom: 4px;
      color: #4b5563;
    }
    .contact-box a {
      color: var(--red);
      text-decoration: none;
    }

    /* Footer */
    .footer {
      background: var(--navy);
      color: rgba(255,255,255,0.7);
      text-align: center;
      padding: 24px 0;
      font-size: 12px;
    }
    .footer a {
      color: rgba(255,255,255,0.9);
      text-decoration: none;
    }
    .footer .media-note {
      font-style: italic;
      margin-top: 8px;
    }

    /* Responsive */
    @media (max-width: 768px) {
      .container { padding: 0 20px; }
      .header { padding: 28px 0 24px; }
      .header img.logo { height: 70px; max-width: 180px; }
      .header h1 { font-size: 20px; padding: 0 10px; }
      .header .subtitle { font-size: 13px; padding: 0 10px; }
      h2 { font-size: 18px; }
      .content .lead { font-size: 14px; }
      .key-figures { gap: 10px; }
      .key-figure { min-width: 100px; padding: 14px 16px; }
      .key-figure .number { font-size: 22px; }
      .launch-box { padding: 24px 16px; }
      .launch-box h3 { font-size: 18px; }
      .launch-box .launch-date { font-size: 17px; }
      .plans-table { font-size: 12px; }
      .plans-table th, .plans-table td { padding: 8px 10px; }
      .quote-block { padding: 18px 20px; }
      .about-box { padding: 20px; }
      .contact-box { padding: 18px 20px; }
    }

    @media (max-width: 480px) {
      .container { padding: 0 14px; }
      .header { padding: 20px 0 18px; }
      .header img.logo { height: 56px; max-width: 150px; }
      .header h1 { font-size: 17px; }
      .header .badge { font-size: 9px; padding: 5px 14px; }
      h2 { font-size: 16px; }
      p, .features-list li { font-size: 13px; }
      .key-figures { flex-direction: column; }
      .key-figure { min-width: unset; }
      .plans-table { display: block; overflow-x: auto; white-space: nowrap; }
      .meta-bar { font-size: 11px; }
    }

    @media print {
      body { font-size: 12px; }
      .container { max-width: 100%; padding: 0 24px; }
      .header { padding: 24px 0 20px; }
      .header h1 { font-size: 20px; }
    }
  </style>
</head>
<body>

  <!-- Header -->
  <div class="header">
    <div class="container">
      <img class="logo" src="data:image/png;base64,${logoWhiteBase64}" alt="LawCodex" />
      <div class="badge">Communiqué de presse</div>
      <h1>LawCodex lance officiellement sa plateforme d'intelligence artificielle juridique en Suisse le 5 mars 2026</h1>
      <p class="subtitle">La première plateforme suisse qui allie IA et droit pour rendre la justice accessible à tous — particuliers, professionnels et entreprises.</p>
    </div>
  </div>

  <!-- Meta bar -->
  <div class="meta-bar">
    <div class="container">
      <strong>Pour une diffusion — Le jeudi 5 mars 2026 à 12h30</strong> &nbsp;|&nbsp; Suisse, le 3 mars 2026
    </div>
  </div>

  <!-- Content -->
  <div class="container content">

    <p class="lead">
      <strong>Suisse, le 3 mars 2026</strong> — LawCodex, plateforme SaaS d'intelligence artificielle dédiée au droit, annonce son lancement officiel le <strong>mercredi 5 mars 2026 à 12h30</strong>. Conçue en Suisse, la plateforme ambitionne de transformer radicalement l'accès au droit en combinant des technologies de pointe — recherche sémantique, analyse prédictive, traitement du langage naturel — avec une couverture juridique couvrant la Suisse, la France, l'Union européenne et plusieurs juridictions internationales.
    </p>

    <!-- Story -->
    <h2>Une plateforme née d'un combat personnel</h2>
    <p>L'histoire de LawCodex est profondément humaine. Son fondateur, <strong>Steven Hori</strong>, n'est pas juriste. C'est un père. En 2021, confronté à l'enlèvement international de sa fille depuis la Suisse vers l'étranger, il s'est retrouvé seul face à un système judiciaire complexe et intimidant, devant mener un combat juridique entre plusieurs pays, pour plus d'infos : <a href="https://bringkimhome.ch/" target="_blank" style="color:${COLORS.darkRed}">bringkimhome.ch</a>.</p>
    <p>De cette épreuve est née une conviction : <strong>le droit ne devrait jamais être un privilège réservé à ceux qui ont les moyens de s'offrir les meilleurs avocats</strong>. Steven Hori a d'abord construit un outil pour lui-même, pour comprendre et maîtriser le droit au lieu de le subir. Puis il a décidé de mettre cette puissance à la portée de tous.</p>

    <div class="quote-block">
      <p class="quote-text">« La justice est un droit accessible à toutes et à tous. LawCodex est né de cette conviction. Notre mission est de transformer l'incertitude juridique en stratégie claire et de redonner le contrôle aux citoyens. »</p>
      <p class="quote-author">— Steven Hori, Fondateur de LawCodex</p>
    </div>

    <!-- Features -->
    <h2>Des fonctionnalités inédites au service du droit</h2>
    <p>LawCodex se distingue par une suite complète d'outils alimentés par l'intelligence artificielle :</p>
    <ul class="features-list">
      <li><strong>Recherche sémantique intelligente</strong> — Un moteur qui comprend le sens des questions juridiques et retrouve les textes pertinents parmi des milliers de sources (lois fédérales suisses, jurisprudence, codes européens).</li>
      <li><strong>Assistant IA juridique</strong> — Des réponses instantanées aux questions de droit, avec citation systématique des sources légales.</li>
      <li><strong>IA stratégique et prédictive</strong> — Analyse de cas, simulation de défense, anticipation des issues juridiques grâce au machine learning.</li>
      <li><strong>Génération automatique de documents</strong> — Courriers juridiques, mises en demeure, contrats, adaptés à chaque situation.</li>
      <li><strong>Gestion électronique de documents (GED)</strong> — Organisation, annotation, versioning et partage sécurisé de l'ensemble des pièces d'un dossier.</li>
      <li><strong>OCR avancé</strong> — Numérisation et indexation automatique de documents papier.</li>
      <li><strong>Couverture multi-juridictions</strong> — Suisse, et bientôt France, Belgique, Allemagne, Espagne, Royaume-Uni, États-Unis, Canada, Brésil et Union européenne.</li>
      <li><strong>Multilingue</strong> — Interface disponible en 7 langues : français, allemand, anglais, italien, espagnol, portugais et néerlandais.</li>
    </ul>

    <!-- Plans -->
    <h2>Un modèle accessible à tous les profils</h2>
    <p>LawCodex s'adresse aussi bien aux <strong>particuliers</strong> confrontés à des démarches juridiques qu'aux <strong>avocats</strong>, <strong>juristes d'entreprise</strong>, <strong>étudiants en droit</strong> et <strong>entrepreneurs</strong>.</p>
    <table class="plans-table">
      <thead>
        <tr>
          <th>Plan</th>
          <th>Public cible</th>
          <th>Points clés</th>
        </tr>
      </thead>
      <tbody>
        <tr>
          <td>Particulier</td>
          <td>Citoyens, étudiants</td>
          <td>Recherche juridique IA, analyse stratégique, génération de documents, GED personnelle</td>
        </tr>
        <tr>
          <td>Professionnel</td>
          <td>Avocats, juristes</td>
          <td>+ CRM intégré, gestion de mandats, agenda avec prise de RDV en ligne, facturation, time tracking</td>
        </tr>
        <tr>
          <td>Entreprise</td>
          <td>Cabinets, sociétés</td>
          <td>+ Multi-utilisateurs, multi-sites, gestion de marque, intégrations tierces, analytics avancés</td>
        </tr>
      </tbody>
    </table>

    <!-- Expansion -->
    <h2>La Suisse comme point de départ, l'Europe comme horizon</h2>
    <p>Basée en Suisse, LawCodex bénéficie d'une intégration native avec les sources juridiques fédérales suisses (Fedlex, jurisprudence du Tribunal fédéral) et s'inscrit dans une stratégie d'expansion européenne. La plateforme a déjà annoncé sa volonté de nouer des <strong>partenariats stratégiques avec plus de 10 cabinets d'avocats suisses</strong> et mène actuellement une <strong>levée de fonds de 500 000 CHF</strong> pour accélérer son développement.</p>

    <!-- Key figures -->
    <h2>Chiffres clés</h2>
    <div class="key-figures">
      <div class="key-figure">
        <span class="number">10+</span>
        <span class="label">Professionnels</span>
      </div>
      <div class="key-figure">
        <span class="number">200+</span>
        <span class="label">Documents juridiques</span>
      </div>
      <div class="key-figure">
        <span class="number">11</span>
        <span class="label">Juridictions</span>
      </div>
      <div class="key-figure">
        <span class="number">7</span>
        <span class="label">Langues</span>
      </div>
      <div class="key-figure">
        <span class="number">10h/j</span>
        <span class="label">Support 5j/7</span>
      </div>
    </div>

    <!-- Launch -->
    <div class="launch-box">
      <h3>Date de lancement</h3>
      <p class="launch-date">Mercredi 5 mars 2026 à 12h30 (CET)</p>
      <p class="launch-url">La plateforme sera accessible à l'adresse : <a href="https://lawcodex.com">https://lawcodex.com</a></p>
    </div>

    <!-- About -->
    <div class="about-box">
      <h3>À propos de LawCodex</h3>
      <p>LawCodex est une plateforme SaaS suisse de recherche et d'analyse juridique propulsée par l'intelligence artificielle. Fondée entre 2022-2025 par Steven Hori, l'entreprise poursuit une mission claire : rendre le droit compréhensible, accessible et actionnable pour tous. En combinant recherche sémantique, IA prédictive, traitement du langage naturel et gestion documentaire avancée, LawCodex transforme la manière dont les particuliers et les professionnels interagissent avec le droit.</p>
      <div class="links">
        <a href="https://lawcodex.com" target="_blank">lawcodex.com</a>
        <a href="https://lawcodexpress.com" target="_blank">lawcodexpress.com</a>
        <a href="https://www.youtube.com/@lawcodex_ai" target="_blank">YouTube</a>
        <a href="https://www.instagram.com/lawCodex" target="_blank">Instagram</a>
      </div>
    </div>

    <!-- Contact -->
    <div class="contact-box">
      <h3>Contact presse</h3>
      <p><strong>Nom :</strong> Emily Hornsby-Martinez</p>
      <p><strong>Rôle :</strong> Co-founder & CEO</p>
      <p><strong>Email :</strong> <a href="mailto:offices@lawcodex.com">offices@lawcodex.com</a></p>
      <p><strong>Téléphone :</strong> <a href="tel:+41215192363">+41 21 519 23 63</a></p>
      <p><strong>Site :</strong> <a href="https://lawcodex.com/press" target="_blank">lawcodex.com/press</a></p>
    </div>

  </div>

  <!-- Footer -->
  <div class="footer">
    <div class="container">
      <p><em>Visuels haute résolution, logos et kit média disponibles sur demande ou à l'adresse :</em></p>
      <p><a href="https://lawcodexpress.com/kit-media/" target="_blank">https://lawcodexpress.com/kit-media/</a></p>
      <p class="media-note" style="margin-top:16px">© 2026 LawCodex — Decode Law & Deploy Strategy</p>
    </div>
  </div>

</body>
</html>`;

  const htmlPath = path.join(DIR, 'CP-LANCEMENT-LAWCODEX-05-MARS-2026.html');
  fs.writeFileSync(htmlPath, html, 'utf-8');
  console.log('✅ HTML créé :', htmlPath);
  return htmlPath;
}

// ────────────────────────────────────────
// 2. PDF via Puppeteer
// ────────────────────────────────────────
async function generatePDF(htmlPath) {
  const browser = await puppeteer.launch({
    headless: true,
    args: ['--no-sandbox', '--disable-setuid-sandbox']
  });
  const page = await browser.newPage();
  const htmlContent = fs.readFileSync(htmlPath, 'utf-8');
  await page.setContent(htmlContent, { waitUntil: 'networkidle0' });
  
  const pdfPath = path.join(DIR, 'CP-LANCEMENT-LAWCODEX-05-MARS-2026.pdf');
  await page.pdf({
    path: pdfPath,
    format: 'A4',
    printBackground: true,
    margin: { top: '0', right: '0', bottom: '0', left: '0' }
  });
  await browser.close();
  console.log('✅ PDF créé :', pdfPath);
}

// ────────────────────────────────────────
// 3. DOCX via docx
// ────────────────────────────────────────
async function generateDOCX() {
  const NAVY = COLORS.navyBlue.replace('#', '');
  const RED = COLORS.darkRed.replace('#', '');
  const GRAY = COLORS.gray.replace('#', '');

  const doc = new Document({
    styles: {
      default: {
        document: {
          run: { font: 'Calibri', size: 22, color: '374151' },
          paragraph: { spacing: { after: 160, line: 320 } }
        },
        heading1: {
          run: { font: 'Georgia', size: 36, bold: true, color: NAVY },
          paragraph: { spacing: { before: 360, after: 200 }, border: { bottom: { style: BorderStyle.SINGLE, size: 6, color: RED } } }
        },
        heading2: {
          run: { font: 'Georgia', size: 28, bold: true, color: NAVY },
          paragraph: { spacing: { before: 320, after: 160 }, border: { bottom: { style: BorderStyle.SINGLE, size: 4, color: RED } } }
        },
      }
    },
    sections: [{
      properties: {
        page: { margin: { top: 1000, bottom: 800, left: 1200, right: 1200 } }
      },
      children: [
        // Logo
        new Paragraph({
          alignment: AlignmentType.CENTER,
          spacing: { after: 200 },
          children: [
            new ImageRun({ data: logoBuffer, transformation: { width: 200, height: 140 }, type: 'png' })
          ]
        }),

        // Badge
        new Paragraph({
          alignment: AlignmentType.CENTER,
          spacing: { after: 100 },
          children: [
            new TextRun({ text: 'COMMUNIQUÉ DE PRESSE', font: 'Calibri', size: 18, bold: true, color: RED, allCaps: true })
          ]
        }),

        // Title
        new Paragraph({
          alignment: AlignmentType.CENTER,
          spacing: { after: 200 },
          children: [
            new TextRun({
              text: "LawCodex lance officiellement sa plateforme d'intelligence artificielle juridique en Suisse le 5 mars 2026",
              font: 'Georgia', size: 32, bold: true, color: NAVY
            })
          ]
        }),

        // Subtitle
        new Paragraph({
          alignment: AlignmentType.CENTER,
          spacing: { after: 300 },
          children: [
            new TextRun({
              text: "La première plateforme suisse qui allie IA et droit pour rendre la justice accessible à tous — particuliers, professionnels et entreprises.",
              font: 'Calibri', size: 22, italics: true, color: GRAY
            })
          ]
        }),

        // Meta bar
        new Paragraph({
          alignment: AlignmentType.CENTER,
          spacing: { after: 300 },
          shading: { type: ShadingType.SOLID, color: 'F3F4F6' },
          border: { bottom: { style: BorderStyle.SINGLE, size: 6, color: RED } },
          children: [
            new TextRun({ text: 'Pour une diffusion — Le jeudi 5 mars 2026 à 12h30', font: 'Calibri', size: 20, bold: true, color: NAVY }),
            new TextRun({ text: '  |  Suisse, le 3 mars 2026', font: 'Calibri', size: 20, color: GRAY })
          ]
        }),

        // Lead
        new Paragraph({
          spacing: { after: 300 },
          border: { bottom: { style: BorderStyle.SINGLE, size: 1, color: 'E5E7EB' } },
          children: [
            new TextRun({ text: 'Suisse, le 3 mars 2026', bold: true }),
            new TextRun({ text: " — LawCodex, plateforme SaaS d'intelligence artificielle dédiée au droit, annonce son lancement officiel le " }),
            new TextRun({ text: 'mercredi 5 mars 2026 à 12h30', bold: true }),
            new TextRun({ text: ". Conçue en Suisse, la plateforme ambitionne de transformer radicalement l'accès au droit en combinant des technologies de pointe — recherche sémantique, analyse prédictive, traitement du langage naturel — avec une couverture juridique couvrant la Suisse, la France, l'Union européenne et plusieurs juridictions internationales." })
          ]
        }),

        // Section: Story
        new Paragraph({ heading: HeadingLevel.HEADING_2, children: [new TextRun({ text: 'Une plateforme née d\'un combat personnel' })] }),
        
        new Paragraph({ children: [
          new TextRun({ text: "L'histoire de LawCodex est profondément humaine. Son fondateur, " }),
          new TextRun({ text: "Steven Hori", bold: true }),
          new TextRun({ text: ", n'est pas juriste. C'est un père. En 2021, confronté à l'enlèvement international de sa fille depuis la Suisse vers l'étranger, il s'est retrouvé seul face à un système judiciaire complexe et intimidant, devant mener un combat juridique entre plusieurs pays, pour plus d'infos : bringkimhome.ch." })
        ]}),

        new Paragraph({ children: [
          new TextRun({ text: "De cette épreuve est née une conviction : " }),
          new TextRun({ text: "le droit ne devrait jamais être un privilège réservé à ceux qui ont les moyens de s'offrir les meilleurs avocats", bold: true }),
          new TextRun({ text: ". Steven Hori a d'abord construit un outil pour lui-même, pour comprendre et maîtriser le droit au lieu de le subir. Puis il a décidé de mettre cette puissance à la portée de tous." })
        ]}),

        // Quote
        new Paragraph({
          spacing: { before: 200, after: 200 },
          indent: { left: 600 },
          border: { left: { style: BorderStyle.SINGLE, size: 12, color: RED } },
          shading: { type: ShadingType.SOLID, color: 'FDF9F6' },
          children: [
            new TextRun({ text: "« La justice est un droit accessible à toutes et à tous. LawCodex est né de cette conviction. Notre mission est de transformer l'incertitude juridique en stratégie claire et de redonner le contrôle aux citoyens. »", font: 'Georgia', italics: true, color: NAVY, size: 22 }),
          ]
        }),
        new Paragraph({
          indent: { left: 600 },
          spacing: { after: 300 },
          children: [
            new TextRun({ text: "— Steven Hori, Fondateur de LawCodex", bold: true, color: RED, size: 20 })
          ]
        }),

        // Section: Features
        new Paragraph({ heading: HeadingLevel.HEADING_2, children: [new TextRun({ text: 'Des fonctionnalités inédites au service du droit' })] }),
        new Paragraph({ children: [new TextRun({ text: "LawCodex se distingue par une suite complète d'outils alimentés par l'intelligence artificielle :" })] }),

        ...[ 
          ['Recherche sémantique intelligente', 'Un moteur qui comprend le sens des questions juridiques et retrouve les textes pertinents parmi des milliers de sources.'],
          ['Assistant IA juridique', 'Des réponses instantanées aux questions de droit, avec citation systématique des sources légales.'],
          ['IA stratégique et prédictive', "Analyse de cas, simulation de défense, anticipation des issues juridiques grâce au machine learning."],
          ['Génération automatique de documents', 'Courriers juridiques, mises en demeure, contrats, adaptés à chaque situation.'],
          ['Gestion électronique de documents (GED)', "Organisation, annotation, versioning et partage sécurisé de l'ensemble des pièces d'un dossier."],
          ['OCR avancé', 'Numérisation et indexation automatique de documents papier.'],
          ['Couverture multi-juridictions', 'Suisse, et bientôt France, Belgique, Allemagne, Espagne, Royaume-Uni, États-Unis, Canada, Brésil et UE.'],
          ['Multilingue', 'Interface disponible en 7 langues : français, allemand, anglais, italien, espagnol, portugais et néerlandais.'],
        ].map(([title, desc]) => new Paragraph({
          spacing: { after: 100 },
          indent: { left: 360 },
          children: [
            new TextRun({ text: '■ ', color: RED, size: 18 }),
            new TextRun({ text: title, bold: true, color: NAVY }),
            new TextRun({ text: ` — ${desc}` })
          ]
        })),

        // Section: Plans table
        new Paragraph({ heading: HeadingLevel.HEADING_2, children: [new TextRun({ text: 'Un modèle accessible à tous les profils' })] }),
        new Paragraph({ children: [
          new TextRun({ text: "LawCodex s'adresse aussi bien aux " }),
          new TextRun({ text: "particuliers", bold: true }),
          new TextRun({ text: " confrontés à des démarches juridiques qu'aux " }),
          new TextRun({ text: "avocats", bold: true }),
          new TextRun({ text: ", " }),
          new TextRun({ text: "juristes d'entreprise", bold: true }),
          new TextRun({ text: ", " }),
          new TextRun({ text: "étudiants en droit", bold: true }),
          new TextRun({ text: " et " }),
          new TextRun({ text: "entrepreneurs", bold: true }),
          new TextRun({ text: "." })
        ]}),

        new Table({
          width: { size: 100, type: WidthType.PERCENTAGE },
          rows: [
            new TableRow({
              tableHeader: true,
              children: ['Plan', 'Public cible', 'Points clés'].map(text => new TableCell({
                shading: { type: ShadingType.SOLID, color: NAVY },
                children: [new Paragraph({ children: [new TextRun({ text, bold: true, color: 'FFFFFF', size: 20 })] })]
              }))
            }),
            ...[
              ['Particulier', 'Citoyens, étudiants', 'Recherche juridique IA, analyse stratégique, génération de documents, GED personnelle'],
              ['Professionnel', 'Avocats, juristes', '+ CRM intégré, gestion de mandats, agenda avec prise de RDV en ligne, facturation, time tracking'],
              ['Entreprise', 'Cabinets, sociétés', '+ Multi-utilisateurs, multi-sites, gestion de marque, intégrations tierces, analytics avancés']
            ].map((cells, i) => new TableRow({
              children: cells.map((text, j) => new TableCell({
                shading: i % 2 === 1 ? { type: ShadingType.SOLID, color: 'F9FAFB' } : undefined,
                children: [new Paragraph({ children: [new TextRun({ text, bold: j === 0, color: j === 0 ? NAVY : '374151', size: 20 })] })]
              }))
            }))
          ]
        }),

        // Section: Europe
        new Paragraph({ heading: HeadingLevel.HEADING_2, spacing: { before: 300 }, children: [new TextRun({ text: 'La Suisse comme point de départ, l\'Europe comme horizon' })] }),
        new Paragraph({ children: [
          new TextRun({ text: "Basée en Suisse, LawCodex bénéficie d'une intégration native avec les sources juridiques fédérales suisses (Fedlex, jurisprudence du Tribunal fédéral) et s'inscrit dans une stratégie d'expansion européenne. La plateforme a déjà annoncé sa volonté de nouer des " }),
          new TextRun({ text: "partenariats stratégiques avec plus de 10 cabinets d'avocats suisses", bold: true }),
          new TextRun({ text: " et mène actuellement une " }),
          new TextRun({ text: "levée de fonds de 500 000 CHF", bold: true }),
          new TextRun({ text: " pour accélérer son développement." })
        ]}),

        // Key figures
        new Paragraph({ heading: HeadingLevel.HEADING_2, children: [new TextRun({ text: 'Chiffres clés' })] }),
        ...[
          ['10+', 'professionnels dans l\'équipe'],
          ['200+', 'modèles de documents juridiques'],
          ['11', 'juridictions couvertes'],
          ['7', 'langues d\'interface'],
          ['10h/j, 5j/7', 'Support']
        ].map(([num, label]) => new Paragraph({
          spacing: { after: 80 },
          indent: { left: 360 },
          children: [
            new TextRun({ text: num, bold: true, color: RED, size: 24, font: 'Georgia' }),
            new TextRun({ text: `  ${label}`, color: GRAY })
          ]
        })),

        // Launch box
        new Paragraph({
          spacing: { before: 400, after: 100 },
          alignment: AlignmentType.CENTER,
          shading: { type: ShadingType.SOLID, color: NAVY },
          children: [
            new TextRun({ text: '\nDate de lancement', font: 'Georgia', size: 28, bold: true, color: 'FFFFFF' })
          ]
        }),
        new Paragraph({
          alignment: AlignmentType.CENTER,
          shading: { type: ShadingType.SOLID, color: NAVY },
          children: [
            new TextRun({ text: 'Mercredi 5 mars 2026 à 12h30 (CET)', size: 26, bold: true, color: 'FBBF24' })
          ]
        }),
        new Paragraph({
          alignment: AlignmentType.CENTER,
          spacing: { after: 300 },
          shading: { type: ShadingType.SOLID, color: NAVY },
          children: [
            new TextRun({ text: 'https://lawcodex.com\n', size: 22, color: '93C5FD' })
          ]
        }),

        // About
        new Paragraph({ heading: HeadingLevel.HEADING_2, children: [new TextRun({ text: 'À propos de LawCodex' })] }),
        new Paragraph({
          shading: { type: ShadingType.SOLID, color: 'FAFAFA' },
          border: { top: { style: BorderStyle.SINGLE, size: 1, color: 'E5E7EB' }, bottom: { style: BorderStyle.SINGLE, size: 1, color: 'E5E7EB' }, left: { style: BorderStyle.SINGLE, size: 1, color: 'E5E7EB' }, right: { style: BorderStyle.SINGLE, size: 1, color: 'E5E7EB' } },
          children: [
            new TextRun({ text: "LawCodex est une plateforme SaaS suisse de recherche et d'analyse juridique propulsée par l'intelligence artificielle. Fondée entre 2022-2025 par Steven Hori, l'entreprise poursuit une mission claire : rendre le droit compréhensible, accessible et actionnable pour tous. En combinant recherche sémantique, IA prédictive, traitement du langage naturel et gestion documentaire avancée, LawCodex transforme la manière dont les particuliers et les professionnels interagissent avec le droit." })
          ]
        }),
        new Paragraph({ spacing: { after: 80 }, children: [
          new TextRun({ text: 'Site web : ', bold: true, color: NAVY }),
          new ExternalHyperlink({ link: 'https://lawcodex.com', children: [new TextRun({ text: 'lawcodex.com', color: RED, underline: {} })] }),
          new TextRun({ text: '   |   Espace presse : ' , bold: true, color: NAVY }),
          new ExternalHyperlink({ link: 'https://lawcodexpress.com', children: [new TextRun({ text: 'lawcodexpress.com', color: RED, underline: {} })] }),
        ]}),
        new Paragraph({ spacing: { after: 80 }, children: [
          new TextRun({ text: 'YouTube : ', bold: true, color: NAVY }),
          new ExternalHyperlink({ link: 'https://www.youtube.com/@lawcodex_ai', children: [new TextRun({ text: '@lawcodex_ai', color: RED, underline: {} })] }),
          new TextRun({ text: '   |   Instagram : ', bold: true, color: NAVY }),
          new ExternalHyperlink({ link: 'https://www.instagram.com/lawCodex', children: [new TextRun({ text: '@lawCodex', color: RED, underline: {} })] }),
        ]}),

        // Contact
        new Paragraph({ heading: HeadingLevel.HEADING_2, children: [new TextRun({ text: 'Contact presse' })] }),
        new Paragraph({
          shading: { type: ShadingType.SOLID, color: 'F3F4F6' },
          spacing: { after: 60 },
          children: [
            new TextRun({ text: 'Nom : ', bold: true, color: NAVY }),
            new TextRun({ text: 'Emily Hornsby-Martinez' })
          ]
        }),
        new Paragraph({
          shading: { type: ShadingType.SOLID, color: 'F3F4F6' },
          spacing: { after: 60 },
          children: [
            new TextRun({ text: 'Rôle : ', bold: true, color: NAVY }),
            new TextRun({ text: 'Co-founder & CEO' })
          ]
        }),
        new Paragraph({
          shading: { type: ShadingType.SOLID, color: 'F3F4F6' },
          spacing: { after: 60 },
          children: [
            new TextRun({ text: 'Email : ', bold: true, color: NAVY }),
            new ExternalHyperlink({ link: 'mailto:offices@lawcodex.com', children: [new TextRun({ text: 'offices@lawcodex.com', color: RED, underline: {} })] })
          ]
        }),
        new Paragraph({
          shading: { type: ShadingType.SOLID, color: 'F3F4F6' },
          spacing: { after: 60 },
          children: [
            new TextRun({ text: 'Téléphone : ', bold: true, color: NAVY }),
            new TextRun({ text: '+41 21 519 23 63' })
          ]
        }),
        new Paragraph({
          shading: { type: ShadingType.SOLID, color: 'F3F4F6' },
          spacing: { after: 300 },
          children: [
            new TextRun({ text: 'Site : ', bold: true, color: NAVY }),
            new ExternalHyperlink({ link: 'https://lawcodex.com/press', children: [new TextRun({ text: 'lawcodex.com/press', color: RED, underline: {} })] })
          ]
        }),

        // Footer
        new Paragraph({
          alignment: AlignmentType.CENTER,
          spacing: { before: 200 },
          children: [
            new TextRun({ text: 'Visuels haute résolution, logos et kit média disponibles sur demande ou à l\'adresse :', italics: true, color: GRAY, size: 20 })
          ]
        }),
        new Paragraph({
          alignment: AlignmentType.CENTER,
          children: [
            new ExternalHyperlink({ link: 'https://lawcodexpress.com/kit-media/', children: [new TextRun({ text: 'https://lawcodexpress.com/kit-media/', color: RED, underline: {}, size: 20 })] })
          ]
        }),
        new Paragraph({
          alignment: AlignmentType.CENTER,
          spacing: { before: 300 },
          children: [
            new TextRun({ text: '© 2026 LawCodex — Decode Law & Deploy Strategy', color: GRAY, size: 18 })
          ]
        }),
      ]
    }]
  });

  const docxPath = path.join(DIR, 'CP-LANCEMENT-LAWCODEX-05-MARS-2026.docx');
  const buffer = await Packer.toBuffer(doc);
  fs.writeFileSync(docxPath, buffer);
  console.log('✅ DOCX créé :', docxPath);
}

// ────────────────────────────────────────
// Main
// ────────────────────────────────────────
(async () => {
  try {
    console.log('🚀 Génération des fichiers du communiqué de presse LawCodex...\n');
    const htmlPath = generateHTML();
    await generatePDF(htmlPath);
    await generateDOCX();
    console.log('\n✅ Tous les fichiers ont été générés avec succès dans :', DIR);
  } catch (err) {
    console.error('❌ Erreur :', err);
    process.exit(1);
  }
})();
